﻿;============================================
; filename: BootLoader.asm
; chip    : CS8M320
; author  :
; date    : 2023-09-21
;============================================
include "CS8M320.inc"
include "constant_define.inc"
include "ram_define.inc"


;============================================
; program start
;============================================ 
CSCC_RES_VECT   .section rom,addr=0x00   
    goto     Prog_Start         ; go to main program
    goto     Prog_Start
    goto     Prog_Start
    goto     Prog_Start
.ends

;============================================
; interrupt vector table
;============================================
CSCC_INT_VECT   .section rom,addr=0x04 
    goto     INT_FUNCTION       ; go to interrupt function
.ends

;============================================
; BootLoader信息 
;============================================ 
CSCC_BT_INFO    .section rom,addr=0x10
    dw      0x0004              ;BootLoader版本号
    dw      0x0000
    dw      0x1C00              ;用户数据存储区起始地址，须为0x20的整数倍
    dw      0x0000
.ends
 
;============================================
;program
;============================================ 
Prog_Start:
    clrwdt
    clrf    BSR                             ;直接和间接寻址访问PAGE0
    bcf     R_System_Flag,B_Int_Flag        ;中断入口标志为0表示BOOT中断，为1表示用户中断
    call    F_Sys_Init
    movlw   55h
    xorwf   R_APToBoot0,0                   ;判断是否由AP跳入BootLoader
    btfss   STATUS,Z
    goto    L_Reset_Normal
    movlw   0AAh
    xorwf   R_APToBoot1,0             
    btfsc   STATUS,Z
    goto    L_Reset_IAP

L_Reset_Normal:
    call    F_RAM_Clear                     ;正常复位则清RAM后超时跳出
    goto    main

L_Reset_IAP:
    call    F_RAM_Clear
    bsf     R_System_Flag,B_Booting_Flag    ;AP跳回BootLoader则需更长的超时时间
    goto    main

;============================================
; 主程序 
;============================================ 
main:
    clrwdt
    btfss   R_System_Flag,B_RecData_OK              ;判断是否接收到指令帧
    goto    L_TimeOut_Judge
;   bcf     R_System_Flag,B_RecData_OK
    movlw   COMM_CMD_HEAD
    movwf   R_SendHead                              ;接收到指令帧后准备进行回复，先赋值帧头
    bcf     R_System_Flag,B_APIntegrity_Fail        ;接收到指令帧后允许对AP重新校验
    bcf     R_Time_Flag,B_Comm_Start
    bsf     R_System_Flag,B_Booting_Flag

    clrf    R_Boot_TimeOut_Count1
    clrf    R_Boot_TimeOut_Count2

    btfsc   R_System_Flag,B_Comm_Abnormal           ;优先处理通信异常，在延时复位后恢复数据处理
    goto    L_Comm_Abnormal_Judge
    bcf     R_System_Flag,B_RecData_OK
    movfw   R_SendACK
    xorlw   REPLY_CMD_ACK
    btfss   STATUS,Z
    goto    L_Boot_Execute_Fail                     ;帧校验失败则直接进行回复
;   call    F_Decode
;   btfss   R_System_Flag,B_Execute_OK              ;判断解密是否成功
;   goto    L_Boot_Execute_Fail

;============================================
; 用户数据存数区起始地址校验 
;============================================ 
F_User_Data_Start_Check:
    movlw   12h
    movwf   R_ROM_AddrL
    clrf    R_ROM_AddrH
    call    F_ISP_Read
    movfw   R_ROM_DataL
    andlw   0b00011111                          ;地址低位必须为20h的整数倍
    btfss   STATUS,Z
    goto    L_Boot_Execute_Fail

    movfw   R_ROM_DataH
    movwf   R_User_Data_StartH
    movfw   R_ROM_DataL
    movwf   R_User_Data_StartL

;============================================
; 获取查表执行入口 
;============================================ 
L_Table_Addr_Convert:
    movfw   R_RecCommand
    movwf   R_RecCmd_Table
    sublw   20h
    btfsc   STATUS,C
    clrf    R_RecCmd_Table                          ;不支持小于21h的指令
    movfw   R_RecCommand
    sublw   28h
    btfss   STATUS,C
    clrf    R_RecCmd_Table                          ;不支持高于28h的指令
    movlw   0b00001111
    andwf   R_RecCmd_Table,0

L_Command_Execute:
    addpcw
    goto    L_NoSupport
    goto    L_GetVersion                            ;0x21
;   goto    L_PasswordCheck
    goto    L_NoSupport
    goto    L_WriteMemory                           ;0x23
    goto    L_ReadMemory                            ;0x24
;   goto    L_EraseSector
    goto    L_NoSupport
    goto    L_EraseChip                             ;0x26
    goto    L_NoSupport
    goto    L_ROMCheck                              ;0x28

;============================================
; 命令执行函数 
;============================================ 
/*****************命令不支持****************/
L_NoSupport:
    movlw   01h
    movwf   R_SendLength
    movlw   EXECUTE_CMD_NOSUPPORT               ;回复“命令不支持”
    movwf   R_SendExecute
    goto    L_Reply_Execute

/*****************获取版本号****************/
L_GetVersion:
    call    F_Version_Read                      ;读取版本号
    btfss   R_System_Flag,B_Execute_OK          ;判断读取是否成功
    goto    L_Boot_Execute_Fail
    movlw   07h
    movwf   R_SendLength
    movlw   EXECUTE_CMD_SUCCESS                 ;回复“命令执行成功”
    movwf   R_SendExecute
    goto    L_Reply_Execute
    
/******************校验秘钥****************/
;L_PasswordCheck:
;   movlw   11h
;   movwf   R_ROM_AddrL
;   clrf    R_ROM_AddrH
;   call    F_ISP_Read                          ;读取本地秘钥
;
;   movfwl  103h
;   xorwf   R_ROM_DataL,0                       ;校验秘钥低位
;   btfss   Z
;   goto    L_Boot_Execute_Fail
;   movfwl  104h
;   xorwf   R_ROM_DataH,0                       ;校验秘钥高位
;   btfss   Z
;   goto    L_Boot_Execute_Fail
;   bsf     R_ISP_Flag,B_PassWord_Match         ;校验成功则置位标志位，允许操作FLASH
;   movlw   01h
;   movwfl  R_SendLength
;   movlw   EXECUTE_CMD_SUCCESS                 ;回复“命令执行成功”
;   movwfl  R_SendExecute
;   goto    L_Reply_Execute
    
/******************写ROM区****************/
L_WriteMemory:
;   btfsc   R_ISP_Flag,B_PassWord_Match         ;判断秘钥是否匹配
;   goto    L_Write_Area_Judge
;   movlw   01h
;   movwfl  R_SendLength
;   movlw   EXECUTE_CMD_NOACCESS                ;不匹配则回复“命令无权限”
;   movwfl  R_SendExecute
;   goto    L_Reply_Execute
    
;L_Write_Area_Judge:
;   movfwl  R_RecLength
;   movwf   R_ISP_Round_Count
;   movlw   07h
;   subwf   R_ISP_Round_Count,1
;   btfss   C
;   goto    L_Write_ParaErro
;   rr      R_ISP_Round_Count,1                 ;获取写轮次总数

    movfw   R_RecLength                         ;写入数据长度必须为一页（64字节）
    xorlw   45h
    btfss   STATUS,Z
    goto    L_Write_ParaErro
    
    movfw   103h
    movwf   R_ROM_AddrL
    andlw   0b00011111                          ;地址低位必须为20h的整数倍
    btfss   STATUS,Z
    goto    L_Write_ParaErro

    movfw   104h
    movwf   R_ROM_AddrH
    sublw   BOOT_END_ADDRH                      ;不允许写BootLoader区域
    btfsc   STATUS,C
    goto    L_Write_ParaErro

    movlw   AP_END_ADDRL                        ;判断写地址是否超出ROM区，超出则直接回复写ROM失败
    subwf   R_ROM_AddrL,0
    movlw   AP_END_ADDRH
    subwfc  R_ROM_AddrH,0
    btfsc   STATUS,C                            ;高位>1F时，C = 1，执行错误
    goto    L_Write_ParaErro

    movfw   R_User_Data_StartL                  ;判断写地址是否超出AP程序
    subwf   R_ROM_AddrL,0                       ;当R_ROM_AddrL地址的值< AP信息起始地址低位 时，C = 0;否则C= 1
    movfw   R_User_Data_StartH                  ;在现有C = 0时，需R_ROM_AddrH地址的值>=（AP信息起始地址高位 + 1），新C才为0
    subwfc  R_ROM_AddrH,0                       ;在现有C = 1时，需R_ROM_AddrH地址的值>=（AP信息起始地址高位），新C才为0
    btfsc   STATUS,C
    goto    L_Write_Judge

L_Write_ISP:
    call    F_ISP_Write                         ;自动从缓存区读取数据写入ROM，校验写入结果，不成功则重复写3次
    btfss   R_System_Flag,B_Execute_OK
    goto    L_Boot_Execute_Fail
    movlw   01h
    movwf   R_SendLength
    movlw   EXECUTE_CMD_SUCCESS                 ;写入完成则回复“命令执行成功”
    movwf   R_SendExecute
    goto    L_Reply_Execute

L_Write_Judge:
    movlw   1Fh
    xorwf   R_ROM_AddrH,0
    btfss   STATUS,Z                            ;R_ROM_AddrH地址数据为1F，Z=1，跳转
    goto    L_Write_None
    movlw   0E0h
    xorwf   R_ROM_AddrL,0
    btfss   STATUS,Z                            ;R_ROM_AddrL地址数据为E0，Z=1，跳转
    goto    L_Write_None
    goto    L_Write_ISP

L_Write_None:
    movlw   01h
    movwf   R_SendLength
    movlw   EXECUTE_CMD_SUCCESS                 ;写入完成则回复“命令执行成功”
    movwf   R_SendExecute
    goto    L_Reply_Execute

L_Write_ParaErro:
    movlw   01h
    movwf   R_SendLength
    movlw   EXECUTE_CMD_PARA_ERROR              ;地址超限则回复“命令参数错误”
    movwf   R_SendExecute
    goto    L_Reply_Execute

/******************读ROM区****************/
L_ReadMemory:
    movfw   103h
    movwf   R_ROM_AddrL
    movfw   104h
    movwf   R_ROM_AddrH
;   sublw   BOOT_END_ADDRH                      ;不允许写BootLoader区域
;   btfsc   STATUS,C
;   goto    L_Write_ParaErro
    
    movlw   AP_END_ADDRL                        ;判断写地址是否超出ROM区，超出则直接回复读ROM失败
    subwf   R_ROM_AddrL,0
    movlw   AP_END_ADDRH
    subwfc  R_ROM_AddrH,0
    btfsc   STATUS,C
    goto    L_Write_ParaErro

    call    F_ISP_APROM_Read                    ;读取数据，长度最长为64字节
    btfss   R_System_Flag,B_Execute_OK
    goto    L_Boot_Execute_Fail
    movlw   EXECUTE_CMD_SUCCESS                 ;读取完成则回复“命令执行成功”
    movwf   R_SendExecute
    goto    L_Reply_Execute
    
/******************FLASH片擦除****************/
;L_EraseSector:
;   btfsc   R_ISP_Flag,B_PassWord_Match         ;判断秘钥是否匹配
;   goto    L_Erase_Area_Judge
;   movlw   01h
;   movwfl  R_SendLength
;   movlw   EXECUTE_CMD_NOACCESS                ;不匹配则回复“命令无权限”
;   movwfl  R_SendExecute
;   goto    L_Reply_Execute
;
;L_Erase_Area_Judge:
;   movfwl  103h
;   movwf   R_ROM_AddrL
;   movfwl  104h
;   movwf   R_ROM_AddrH
;   sublw   BOOT_END_ADDRH                      ;不允许擦除BootLoader区域
;   btfsc   C
;   goto    L_Erase_ParaErro
;   movfw   R_ROM_AddrH
;   sublw   7Fh                                 ;不允许擦除超出AP ROM区域
;   btfss   C
;   goto    L_Erase_ParaErro
;   
;   call    F_FLASH_Change_Sign                 ;擦写FLASH前需清除完整性标志位，置位FLASH擦写标志位
;   btfss   R_System_Flag,B_Execute_OK
;   goto    L_Boot_Execute_Fail
;   call    F_ISP_Sector_Erase
;   btfss   R_System_Flag,B_Execute_OK
;   goto    L_Boot_Execute_Fail
;   movlw   01h
;   movwfl  R_SendLength
;   movlw   EXECUTE_CMD_SUCCESS                 ;擦除完成则回复“命令执行成功”
;   movwfl  R_SendExecute
;   goto    L_Reply_Execute

;L_Erase_ParaErro:
;   movlw   01h
;   movwfl  R_SendLength
;   movlw   EXECUTE_CMD_PARA_ERROR              ;地址超限则回复“命令参数错误”
;   movwfl  R_SendExecute
;   goto    L_Reply_Execute
    
/******************AP ROM擦除****************/
L_EraseChip:
;   btfsc   R_ISP_Flag,B_PassWord_Match         ;判断秘钥是否匹配
;   goto    L_Erase_Chip_Judge
;   movlw   01h
;   movwfl  R_SendLength
;   movlw   EXECUTE_CMD_NOACCESS                ;不匹配则回复“命令无权限”
;   movwfl  R_SendExecute
;   goto    L_Reply_Execute
;
;L_Erase_Chip_Judge:
;   call    F_FLASH_Change_Sign                 ;擦写FLASH前需清除完整性标志位，置位FLASH擦写标志位
;   btfss   R_System_Flag,B_Execute_OK
;   goto    L_Boot_Execute_Fail
    call    F_ISP_APROM_Erase
    btfss   R_System_Flag,B_Execute_OK
    goto    L_Boot_Execute_Fail
    movlw   01h
    movwf   R_SendLength
    movlw   EXECUTE_CMD_SUCCESS                 ;擦除成功则回复“命令执行成功”
    movwf   R_SendExecute
    goto    L_Reply_Execute
    
/******************AP ROM校验****************/
L_ROMCheck:
    clrf    R_ROMCheck_Temp
    clrf    R_ROM_AddrL
    movlw   AP_START_ADDRH
    movwf   R_ROM_AddrH

L_ROMCheck_Continue:
    call    F_ISP_Read                          ;读取AP ROM数据
    movfw   R_ROM_DataH
    addwf   R_ROMCheck_Temp,1
    movfw   R_ROM_DataL
    addwf   R_ROMCheck_Temp,1                   ;累加AP ROM数据
;   movlw   01h
;   addwf   R_ROM_AddrL,1
;   btfss   STATUS,C
;   goto    L_ROMCheck_Continue
;   incf    R_ROM_AddrH,1
;   movlw   20h
;   subwf   R_ROM_AddrH,0
;   btfss   STATUS,C
;   goto    L_ROMCheck_Continue
    ;incf   R_ROM_AddrL,1
    movlw   01h
    addwf   R_ROM_AddrL,1
    btfss   STATUS,C
    goto    L_ROM_Addr_CheckAP
    incf    R_ROM_AddrH,1
    goto    L_ROM_Addr_CheckAP

L_ROM_Addr_CheckAP:
    movfw   R_User_Data_StartL
    subwf   R_ROM_AddrL,0                       ;当R_ROM_AddrL地址的值< AP信息起始地址低位 时，C = 0;否则C= 1
    movfw   R_User_Data_StartH                  ;在现有C = 0时，需R_ROM_AddrH地址的值>=（AP信息起始地址高位 + 1），新C才为0
    subwfc  R_ROM_AddrH,0                       ;在现有C = 1时，需R_ROM_AddrH地址的值>=（AP信息起始地址高位），新C才为0
    btfss   STATUS,C
    goto    L_ROMCheck_Continue
    goto    L_ROM_Addr_CheckRev
    
L_ROM_Addr_CheckRev:
    movlw   0E0h
    subwf   R_ROM_AddrL,0
    movlw   1Fh
    subwfc  R_ROM_AddrH,0
    btfss   STATUS,C
    goto    L_ROMCheck_AddFF                    ;加FF

    bsf     R_System_Flag,B_APIntegrity_Fail      
    movfw   103h
    xorwf   R_ROMCheck_Temp,0                   ;校验AP ROM区数据
    btfss   STATUS,Z
    goto    L_Boot_Execute_Fail
    call    F_Write_Checksum                    ;写checksum
    btfss   R_System_Flag,B_Execute_OK
    goto    L_Boot_Execute_Fail
    bcf     R_System_Flag,B_APIntegrity_Fail
    bcf     R_System_Flag,B_Booting_Flag
    movlw   01h
    movwf   R_SendLength
    movlw   EXECUTE_CMD_SUCCESS                 ;校验成功则回复“命令执行成功”
    movwf   R_SendExecute
    goto    L_Reply_Execute 

L_ROMCheck_AddFF:
    movlw   0FFh
    addwf   R_ROMCheck_Temp,1
    movlw   0FFh
    addwf   R_ROMCheck_Temp,1
    
    movlw   01h
    addwf   R_ROM_AddrL,1
    btfss   STATUS,C
    goto    L_ROM_Addr_CheckAP
    incf    R_ROM_AddrH,1
    goto    L_ROM_Addr_CheckAP

L_Boot_Execute_Fail:
    movlw   01h
    movwf   R_SendLength
    movlw   EXECUTE_CMD_FAIL                    ;回复“命令执行失败”
    movwf   R_SendExecute

L_Reply_Execute:
;   call    F_Encrypt                           ;加密数据区
;   btfss   R_System_Flag,B_Execute_OK
;   goto    L_Boot_Execute_Fail
    movfw   R_SendLength
    addlw   04h
    movwf   R_FrameCS_Position                  ;确定帧校验码位置
    clrf    R_SendData_Count
    
    clrf    UR0_CR1
    bcf     UR0_ST,UR_SWAP
    movlw   0b00000101
    movwf   UR0_CR1
    clrf    R_SendData_Count
    movfw   R_SendACK
    movwf   R_FrameCS_Calculate                 ;为帧校验码赋初值
    movwf   UR0_TX_REG                          ;开始回复数据
    bsf     UR0_INTE,UR0_TEIE
    
;============================================
; 计时及跳转函数 
;============================================   
L_TimeOut_Judge:
    btfsc   R_System_Flag,B_APIntegrity_Fail        ;AP完整性校验失败则不再进行超时计数，直至重新接收到指令
    goto    main
    btfss   R_Time_Flag,B_Time_1ms
    goto    main
    bcf     R_Time_Flag,B_Time_1ms                  ;等待接收指令时间 1ms，超时后进入通信异常判断
    
/*****************通信超时计时***************/
L_Comm_TimeOut_Judge:
    btfsc   R_Time_Flag,B_Comm_Start                ;判断是否开启通信超时计时功能
    goto    L_Comm_TimeOut_Count                    ;开启了通信超时计时功能，跳到通信超时计时
    clrf    R_Comm_TimeOut_Count                    ;没有开启通信超时计时功能，跳到通信异常判断
    goto    L_Comm_Abnormal_Judge
    
L_Comm_TimeOut_Count:
    incf    R_Comm_TimeOut_Count,1
    movlw   COMM_TIMEOUT_LIMIT
    subwf   R_Comm_TimeOut_Count,0                  ;判断通信是否超时
    btfss   STATUS,C
    goto    L_Comm_Abnormal_Judge
    bcf     UR0_CR1,RX_EN                           ;超时后先关闭接收功能
    bcf     R_Time_Flag,B_Comm_Start
    clrf    R_RecData_Count                         ;清零接收计数
    clrf    R_Comm_TimeOut_Count                    ;清零超时计数
    nop
    nop
    nop
    nop
    bsf     UR0_CR1,RX_EN                           ;延时一小段时间后再打开接收功能，以复位UART接收FIFO

/*****************异常复位计时***************/
L_Comm_Abnormal_Judge:
    btfsc   R_System_Flag,B_Comm_Abnormal           ;判断是否开启接收异常复位计时功能
    goto    L_Comm_Abnormal_Count                   ;开启了接收异常复位计时功能。跳转，计时15ms
    clrf    R_Comm_Abnormal_Count                   ;没有开启接收异常复位计时功能, 跳转,回main
    goto    L_Boot_TimeOut_Judge
    
L_Comm_Abnormal_Count:
    incf    R_Comm_Abnormal_Count,1
    movlw   COMM_ABNORMAL_RESET_DELAY
    subwf   R_Comm_Abnormal_Count,0                 ;判断是否达到复位时间
    btfss   STATUS,C
    goto    L_Boot_TimeOut_Judge
    bcf     R_System_Flag,B_Comm_Abnormal
    clrf    R_Comm_Abnormal_Count                   ;达到复位时间后清零复位计时
    clrf    R_RecData_Count
    clrf    R_SendData_Count
    bsf     UR0_CR1,RX_EN                           ;开启UART接收功能
    
/*****************跳转超时计时***************/
L_Boot_TimeOut_Judge:
    btfsc   R_System_Flag,B_Booting_Flag            ;判断是否接收过上位机数据
    goto    L_Boot_TimeOut_Count                    ;接收过上位机数据，Boot超时计时
    clrf    R_Boot_TimeOut_Count1
    clrf    R_Boot_TimeOut_Count2
    
L_Reset_TimeOut_Count:
    incf    R_Reset_TimeOut_Count,1
    movlw   RESET_TIMEOUT_LIMIT                     ;非升级复位超时时间 30ms
    subwf   R_Reset_TimeOut_Count,0
    btfss   STATUS,C
    goto    main
    clrf    R_Reset_TimeOut_Count
    call    F_Jump_Judge                            ;非升级复位超时后检查AP完整性
    btfsc   R_System_Flag,B_APIntegrity_Fail        ;AP完整则跳转至AP程序
    goto    main
    call    F_Junp_Init
    goto    0400h
;   goto    main

L_Boot_TimeOut_Count:
    clrf    R_Reset_TimeOut_Count
    incf    R_Boot_TimeOut_Count1,1
    movlw   BOOT_TIMEOUT_LIMIT1
    subwf   R_Boot_TimeOut_Count1,0
    btfss   STATUS,C
    goto    main
    clrf    R_Boot_TimeOut_Count1
    incf    R_Boot_TimeOut_Count2,1
    movlw   BOOT_TIMEOUT_LIMIT2                     ;Boot超时时间4s
    subwf   R_Boot_TimeOut_Count2,0
    btfss   STATUS,C
    goto    main
    clrf    R_Boot_TimeOut_Count2
;   bcf     R_ISP_Flag,B_PassWord_Match             ;超时后需重新进行秘钥匹配
    call    F_Jump_Judge
    btfsc   R_System_Flag,B_APIntegrity_Fail        ;AP完整则跳转至AP程序
    goto    main
    call    F_Junp_Init
    goto    0400h
;   goto    main

.end